//
//  LifeExpectancyViewModel.swift
//  death_app Watch App
//
//  Created by Task Master on 2025-09-16.
//

import SwiftUI
import Combine
import Foundation

@MainActor
class LifeExpectancyViewModel: ObservableObject {
    @Published var predictedDeathAge: Int = 78
    @Published var yearsRemaining: Int = 0
    @Published var monthsRemaining: Int = 0
    @Published var daysRemaining: Int = 0
    @Published var dailyChangeInDays: Double = 0.0
    @Published var healthChangeState: HealthChangeState = .neutral
    @Published var changeSeverity: Double = 0.0
    
    private let predictionService = PredictionService.shared
    private let healthKitService = HealthKitService()
    private var cancellables = Set<AnyCancellable>()
    
    // Real birth date from HealthKit
    private var birthDate: Date?
    
    init() {
        setupServices()
        calculateInitialValues()
    }
    
    func startTracking() {
        updateCountdown()
        
        // Set up periodic updates
        Timer.publish(every: 3600, on: .main, in: .common) // Update every hour
            .autoconnect()
            .sink { [weak self] _ in
                self?.updateLifeExpectancy()
            }
            .store(in: &cancellables)
    }
    
    func updateCountdown() {
        guard let birthDate = birthDate else {
            // Use default values if birth date is not available
            yearsRemaining = predictedDeathAge - 35 // Assume 35 years old
            monthsRemaining = 0
            daysRemaining = 0
            return
        }
        
        let calendar = Calendar.current
        let now = Date()
        
        // Calculate predicted death date
        let predictedDeathDate = calendar.date(byAdding: .year, value: predictedDeathAge, to: birthDate) ?? now
        
        // Calculate remaining time
        let components = calendar.dateComponents([.year, .month, .day], from: now, to: predictedDeathDate)
        
        yearsRemaining = max(0, components.year ?? 0)
        monthsRemaining = max(0, components.month ?? 0)
        daysRemaining = max(0, components.day ?? 0)
    }
    
    private func setupServices() {
        // Subscribe to prediction changes
        predictionService.$currentPrediction
            .debounce(for: .seconds(1), scheduler: RunLoop.main)
            .sink { [weak self] prediction in
                self?.processPredictionUpdate(prediction)
            }
            .store(in: &cancellables)
        
        // Subscribe to health data changes
        healthKitService.$currentHealthData
            .debounce(for: .seconds(2), scheduler: RunLoop.main)
            .sink { [weak self] healthData in
                self?.processHealthDataUpdate(healthData)
            }
            .store(in: &cancellables)
    }
    
    private func calculateInitialValues() {
        Task {
            do {
                // Get birth date from HealthKit
                let healthSnapshot = try await healthKitService.fetchLatestHealthData()
                birthDate = healthSnapshot.dateOfBirth
                
                // Trigger initial prediction calculation
                await predictionService.calculateLifeExpectancy()
                
                await MainActor.run {
                    updateCountdown()
                }
            } catch {
                print("Failed to get initial health data: \(error)")
                // Fallback to default values
                await MainActor.run {
                    updateCountdown()
                }
            }
        }
    }
    
    private func updateLifeExpectancy() {
        Task {
            await predictionService.calculateLifeExpectancy()
        }
    }
    
    private func processPredictionUpdate(_ prediction: LifeExpectancyPrediction?) {
        guard let prediction = prediction else { return }
        
        let previousPrediction = Double(predictedDeathAge)
        let newPrediction = prediction.lifeExpectancy
        let change = newPrediction - previousPrediction
        
        predictedDeathAge = Int(newPrediction)
        
        // Calculate daily change from prediction change
        if let changeFromPrevious = prediction.changeFromPrevious {
            dailyChangeInDays = changeFromPrevious.changeDays / 365.25 // Convert to daily change
        } else {
            dailyChangeInDays = change * 365.25 / 365.25 // Normalize to daily change
        }
        
        updateHealthChangeState(change: Int(change))
        updateCountdown()
    }
    
    private func processHealthDataUpdate(_ healthData: HealthDataSnapshot?) {
        guard let healthData = healthData else { return }
        
        // Update birth date if available
        if let dateOfBirth = healthData.dateOfBirth {
            birthDate = dateOfBirth
        }
        
        // Trigger new prediction calculation when health data changes
        Task {
            await predictionService.calculateLifeExpectancy()
        }
    }
    
    private func processHealthMetrics(_ metrics: HealthMetrics) {
        // Process real-time health changes and provide immediate feedback
        let impactScore = calculateHealthImpact(from: metrics)
        
        if abs(impactScore) > 0.1 {
            dailyChangeInDays = impactScore
            changeSeverity = min(abs(impactScore) / 2.0, 1.0)
            
            if impactScore > 0 {
                healthChangeState = .positive
            } else {
                healthChangeState = .negative
            }
        } else {
            healthChangeState = .neutral
            changeSeverity = 0.0
        }
    }
    
    private func updateHealthChangeState(change: Int) {
        if change > 0 {
            healthChangeState = .positive
            changeSeverity = min(Double(change) / 5.0, 1.0)
        } else if change < 0 {
            healthChangeState = .negative
            changeSeverity = min(Double(abs(change)) / 5.0, 1.0)
        } else {
            healthChangeState = .neutral
            changeSeverity = 0.0
        }
    }
    
    private func calculateHealthImpact(from metrics: HealthMetrics) -> Double {
        // Simplified health impact calculation
        var impact: Double = 0.0
        
        // Heart rate impact
        if let heartRate = metrics.restingHeartRate {
            let optimalRange = 60...100
            if optimalRange.contains(heartRate) {
                impact += 0.1
            } else {
                impact -= 0.2
            }
        }
        
        // Steps impact
        if let steps = metrics.stepCount {
            if steps >= 8000 {
                impact += 0.2
            } else if steps < 3000 {
                impact -= 0.1
            }
        }
        
        // Sleep impact
        if let sleepHours = metrics.sleepHours {
            if 7...9 ~= sleepHours {
                impact += 0.1
            } else {
                impact -= 0.15
            }
        }
        
        return impact
    }
    
    private func getCurrentAge() -> Int {
        guard let birthDate = birthDate else { return 35 }
        let calendar = Calendar.current
        let components = calendar.dateComponents([.year], from: birthDate, to: Date())
        return components.year ?? 35
    }
}